
// Compile with Hi-Tech PICC Lite.
// Example program to demonstrate how we may easily reuse assembler code taken
// from "PIC to PS/2 Mouse and Keyboard Interfacing" (EPE Aug 04).
// Implemented using standard library functions and adding in some shift key processing.

#include <stdio.h>
#include <pic.h>
#include "lcd.h"

// scan codes for left and right shift keys
#define LSHIFT 0x12
#define RSHIFT 0x59
// break code
#define BREAK  0xF0

// Setup the configuration word
__CONFIG(0x3f31);

// forward declaration of the external assembler function
struct scancodes{
	char code1;
	char code2;
};
extern struct scancodes receive(void);

// Subset of the scan codes and ASCII equivalent Set 2, taken from Table 1 of the original article.
const unsigned char codes[][2] = {
#if defined(_16F627)
0x1c,'a',0x32,'b',0x21,'c',0x23,'d',0x24,'e',0x2b,'f',0x34,'g',0x33,'h',0x43,'i',0x3b,'j',
0x42,'k',0x4b,'l',0x3a,'m',0x31,'n',0x44,'o',0x4d,'p',
#else
// assumption is that > 1K program memory available
0x1c,'a',0x32,'b',0x21,'c',0x23,'d',0x24,'e',0x2b,'f',0x34,'g',0x33,'h',0x43,'i',0x3b,'j',
0x42,'k',0x4b,'l',0x3a,'m',0x31,'n',0x44,'o',0x4d,'p',0x15,'q',0x2d,'r',0x1b,'s',0x2c,'t',
0x3c,'u',0x2a,'v',0x1d,'w',0x22,'x',0x35,'y',0x1a,'z',0x45,'0',0x16,'1',0x1e,'2',0x26,'3',
0x25,'4',0x2e,'5',0x36,'6',0x3d,'7',0x3e,'8',0x46,'9',0x0e,'`',0x4e,'-',0x55,'=',0x5d,'\'',
0x54,'[',0x5b,']',0x4c,';',0x52,'\'',0x41,',',0x49,'.',0x4a,'/',
#endif
0,0};

// New table of shifted equavlents for the same keys
const unsigned char shifted_codes[][2] = {
#if defined(_16F627)
0x1c,'A',0x32,'B',0x21,'C',0x23,'D',0x24,'E',0x2b,'F',0x34,'G',0x33,'H',0x43,'I',0x3b,'J',
0x42,'K',0x4b,'L',0x3a,'M',0x31,'N',0x44,'O',0x4d,'P',
#else
// assumption is that > 1K program memory available
0x1c,'A',0x32,'B',0x21,'C',0x23,'D',0x24,'E',0x2b,'F',0x34,'G',0x33,'H',0x43,'I',0x3b,'J',
0x42,'K',0x4b,'L',0x3a,'M',0x31,'N',0x44,'O',0x4d,'P',0x15,'Q',0x2d,'R',0x1b,'S',0x2c,'T',
0x3c,'U',0x2a,'V',0x1d,'W',0x22,'X',0x35,'Y',0x1a,'Z',0x45,')',0x16,'!',0x1e,'"',0x26,'',
0x25,'$',0x2e,'%',0x36,'^',0x3d,'&',0x3e,'*',0x46,'(',0x0e,'',0x4e,'_',0x55,'+',0x5d,'|',
0x54,'{',0x5b,'}',0x4c,':',0x52,'@',0x41,'<',0x49,'>',0x4a,'?',
#endif
0,0};

// Main function
void main(void)
{
	unsigned char key;

	// clear ports
	PORTA = 0;
	PORTB = 0;

	// PORTA as digital
	CMCON = 7;
	// RA0, RA1 as input
	TRISA = 3;
	// PORTB as outputs
	TRISB = 0;

	// Initialise and clear the LCD
	lcd_init();
	lcd_clear();

	printf("Press a key:");
	// Loop forever
	while(1)
	{
		// waiting for a key press and
		// printing the result.
		key = getch();
		lcd_clear();
		printf("You pressed [%c]", key);
	}
}

// Pass straight through to LCD
void putch(char c)
{
	lcd_putch(c);
}


// Simple lookup table access.
unsigned char table_lookup(const unsigned char array[][2], char code)
{
	unsigned char x;

	for(x = 0; array[x][0]; x++)
		if (array[x][0] == code)
		{
			return(array[x][1]);
		}
	// return question mark if the codes not in the table
	return '?';
}

// Gets the scan codes, interprets them using the lookup tables
// and returns the ASCII equivalent of the key-press
unsigned char getch(void)
{
	struct scancodes scan;
	// remember shift key state
	static unsigned char shift = 0;

	do
	{
		// get PS/2 key code
		scan = receive();
		// check for shift keys
		if (scan.code1 == LSHIFT || scan.code1 == RSHIFT)
		{
			// if we get a break code, shift not pressed
			shift = (scan.code2 != BREAK);
		}
		else
			break;
	}while(1);

	// return the correct ASCII value for the scan code
	if(!shift)
		return table_lookup(codes, scan.code1);
	else
		return table_lookup(shifted_codes, scan.code1);
}

